# centroid_nspire_micropy_ascii.py
# Composite centroid calculator (MicroPython-safe, ASCII-only for TI-Nspire)
# - No Unicode characters, no f-strings, no exotic formatting
# - Uses only print, input, math.pi, basic string concatenation

from math import pi

# ---------- Input helpers ----------
def prompt_float(msg):
    while True:
        try:
            return float(input(msg))
        except:
            print("Please enter a valid number.")

def prompt_choice(msg, valid_list):
    # Ask for input until user types one of the valid strings.
    # Empty input counts as '0' if '0' is in valid_list.
    valid = set(valid_list)
    while True:
        s = input(msg).strip()
        if s == "" and ("0" in valid):
            return "0"
        if s in valid:
            return s
        print("Choose one of: " + str(list(valid)))

def prompt_int_in(options, msg):
    # Numeric-only wrapper on prompt_choice
    valid = [str(v) for v in options]
    s = prompt_choice(msg, valid)
    try:
        return int(s)
    except:
        return int(valid[0])

def s(msg):
    return input(msg).strip()

def add(term_list, term):
    term = term.strip()
    if term:
        term_list.append("(" + term + ")")

# ---------- Geometry helpers ----------
def polygon_centroid(vertices):
    n = len(vertices)
    if n < 3:
        raise ValueError("Polygon needs at least 3 vertices.")
    A2 = 0.0
    Cx_num = 0.0
    Cy_num = 0.0
    i = 0
    while i < n:
        x_i, y_i = vertices[i]
        x_j, y_j = vertices[(i+1) % n]
        cross = x_i*y_j - x_j*y_i
        A2 += cross
        Cx_num += (x_i + x_j) * cross
        Cy_num += (y_i + y_j) * cross
        i += 1
    if abs(A2) < 1e-12:
        raise ValueError("Polygon area ~ 0; check vertices/order.")
    A = abs(0.5*A2)
    Cx = Cx_num/(3.0*A2)
    Cy = Cy_num/(3.0*A2)
    return A, Cx, Cy

# ---------- Numeric shapes ----------
def rect_num():
    print("")
    print("Rectangle (reference = LOWER-LEFT corner)")
    w = prompt_float("Width w: ")
    h = prompt_float("Height h: ")
    x0 = prompt_float("Lower-left x0: ")
    y0 = prompt_float("Lower-left y0: ")
    A  = w*h
    cx = x0 + w/2.0
    cy = y0 + h/2.0
    return "Rectangle", A, cx, cy

def right_triangle_num():
    print("")
    print("Right Triangle (legs from RIGHT-ANGLE corner)")
    a = prompt_float("Leg along +x (a): ")
    b = prompt_float("Leg along y (b): ")
    x0 = prompt_float("Right-angle corner x0: ")
    y0 = prompt_float("Right-angle corner y0: ")
    print("Y-leg direction?  1=Up (+y)   2=Down (-y)")
    ydir = prompt_int_in((1,2), "Choose 1/2: ")
    A  = 0.5*a*b
    cx = x0 + a/3.0
    if ydir == 1:
        cy = y0 + b/3.0
        name = "Right Triangle (up)"
    else:
        cy = y0 - b/3.0
        name = "Right Triangle (down)"
    return name, A, cx, cy

def circle_num():
    print("")
    print("Circle (reference = CENTER)")
    r  = prompt_float("Radius r: ")
    xc = prompt_float("Center xc: ")
    yc = prompt_float("Center yc: ")
    A  = pi*r*r
    return "Circle", A, xc, yc

def semicircle_num():
    print("")
    print("Semicircle (reference = CENTER of parent circle)")
    mode = prompt_int_in((1,2), "Enter 1=Radius  or  2=Diameter: ")
    if mode == 1:
        r = prompt_float("Radius r: ")
    else:
        D = prompt_float("Diameter D: ")
        r = D/2.0
    xc = prompt_float("Center xc: ")
    yc = prompt_float("Center yc: ")
    print("Bulge direction?  1=Up  2=Down  3=Right  4=Left")
    ori = prompt_int_in((1,2,3,4), "Choose 1/2/3/4: ")
    A = 0.5*pi*r*r
    d = 4.0*r/(3.0*pi)
    if   ori==1:
        cx,cy = xc,     yc + d; name="Semicircle (up)"
    elif ori==2:
        cx,cy = xc,     yc - d; name="Semicircle (down)"
    elif ori==3:
        cx,cy = xc + d, yc;     name="Semicircle (right)"
    else:
        cx,cy = xc - d, yc;     name="Semicircle (left)"
    return name, A, cx, cy

def quarter_circle_num():
    print("")
    print("Quarter Circle (reference = RIGHT-ANGLE corner of legs)")
    r  = prompt_float("Radius r: ")
    x0 = prompt_float("Corner x0: ")
    y0 = prompt_float("Corner y0: ")
    print("Quadrant relative to the corner?")
    print(" 1=NE  2=NW  3=SW  4=SE")
    q = prompt_int_in((1,2,3,4), "Choose 1/2/3/4: ")
    A = 0.25*pi*r*r
    d = 4.0*r/(3.0*pi)
    if   q==1: cx,cy = x0 + d, y0 + d; name="Quarter Circle (NE)"
    elif q==2: cx,cy = x0 - d, y0 + d; name="Quarter Circle (NW)"
    elif q==3: cx,cy = x0 - d, y0 - d; name="Quarter Circle (SW)"
    else:      cx,cy = x0 + d, y0 - d; name="Quarter Circle (SE)"
    return name, A, cx, cy

def trapezoid_num():
    print("")
    print("Trapezoid (bases horizontal)")
    x0 = prompt_float("Lower-left x0: ")
    y0 = prompt_float("Lower-left y0: ")
    b1 = prompt_float("Bottom base length b1: ")
    b2 = prompt_float("Top base length b2: ")
    h  = prompt_float("Height h: ")
    sft  = prompt_float("Top base left offset s (can be negative): ")
    verts = [(x0, y0),
             (x0 + b1, y0),
             (x0 + sft + b2, y0 + h),
             (x0 + sft, y0 + h)]
    A, cx, cy = polygon_centroid(verts)
    return "Trapezoid", A, cx, cy

def polygon_generic_num():
    print("")
    print("Generic Polygon (enter vertices in order, CW or CCW)")
    n = prompt_int_in(tuple(range(3, 101)), "Number of vertices n (3..100): ")
    verts = []
    i = 1
    while i <= n:
        x = prompt_float("  Vertex " + str(i) + " - x: ")
        y = prompt_float("  Vertex " + str(i) + " - y: ")
        verts.append((x, y))
        i += 1
    A, cx, cy = polygon_centroid(verts)
    return "Polygon", A, cx, cy

def add_shape_numeric():
    print("")
    print("Add a shape (Numeric):")
    print(" 1) Rectangle (w,h at lower-left)")
    print(" 2) Right Triangle (up/down)")
    print(" 3) Circle (radius at center)")
    print(" 4) Semicircle (radius/diameter; choose bulge)")
    print(" 5) Quarter Circle (quadrant from corner)")
    print(" 6) Trapezoid (bases horizontal, offset allowed)")
    print(" 7) Polygon (generic vertex list)")
    k = prompt_int_in((1,2,3,4,5,6,7), "Choose 1-7: ")

    if   k==1: name,A,cx,cy = rect_num()
    elif k==2: name,A,cx,cy = right_triangle_num()
    elif k==3: name,A,cx,cy = circle_num()
    elif k==4: name,A,cx,cy = semicircle_num()
    elif k==5: name,A,cx,cy = quarter_circle_num()
    elif k==6: name,A,cx,cy = trapezoid_num()
    else:      name,A,cx,cy = polygon_generic_num()

    sign = prompt_choice("Solid (+1) or Hole (-1)? Enter 1 or -1: ", ["1","-1"])
    if sign == "-1":
        A = -A
        name = name + " [HOLE]"
    return name, A, cx, cy

def run_exercise_numeric():
    shapes = []
    while True:
        name, A, cx, cy = add_shape_numeric()
        shapes.append((name, A, cx, cy))
        more = prompt_choice("Add another shape? 1=Yes  0=No: ", ["1","0"])
        if more == "0":
            break

    if not shapes:
        print("")
        print("No shapes entered.")
        return

    Atot = 0.0
    Sx   = 0.0
    Sy   = 0.0
    for sh in shapes:
        Ai = sh[1]
        xi = sh[2]
        yi = sh[3]
        Atot += Ai
        Sx   += Ai*xi
        Sy   += Ai*yi

    print("")
    print("# | Shape | Ai | xi | yi | Ai*xi | Ai*yi")
    print("----------------------------------------")
    i = 1
    for sh in shapes:
        name, A, cx, cy = sh
        print(str(i) + " | " + name + " | " + str(A) + " | " + str(cx) + " | " + str(cy) + " | " + str(A*cx) + " | " + str(A*cy))
        i += 1

    print("")
    print("--- Totals ---")
    print("A_total = " + str(Atot))
    print("Sum(A*xi) = " + str(Sx))
    print("Sum(A*yi) = " + str(Sy))

    if abs(Atot) < 1e-12:
        print("")
        print("WARNING: Net area ~ 0; Composite centroid undefined.")
        return

    xbar = Sx/Atot
    ybar = Sy/Atot

    # Step-by-step strings (ASCII only)
    termA  = []
    termAx = []
    termAy = []
    for sh in shapes:
        Ai = sh[1]; xi = sh[2]; yi = sh[3]
        # Show signs without relying on non-ASCII
        Ai_str = ("+" if Ai >= 0 else "") + str(round(Ai, 4))
        termA.append("(" + Ai_str + ")")
        termAx.append("(" + Ai_str + "x" + str(round(xi, 4)) + ")")
        termAy.append("(" + Ai_str + "x" + str(round(yi, 4)) + ")")

    print("")
    print("--- Calculation Steps ---")
    print("sumA = " + " + ".join(termA) + " = " + str(Atot))
    print("sum(A*xi) = " + " + ".join(termAx) + " = " + str(Sx))
    print("sum(A*yi) = " + " + ".join(termAy) + " = " + str(Sy))

    print("")
    print("Using centroid formulas:")
    print("xbar = sum(A*xi) / sumA = " + str(Sx) + " / " + str(Atot) + " = " + str(xbar))
    print("ybar = sum(A*yi) / sumA = " + str(Sy) + " / " + str(Atot) + " = " + str(ybar))

    # Weighted-average display
    wx_terms = []
    wy_terms = []
    for sh in shapes:
        Ai = sh[1]; xi = sh[2]; yi = sh[3]
        wx_terms.append("(" + str(Ai/Atot) + "x" + str(xi) + ")")
        wy_terms.append("(" + str(Ai/Atot) + "x" + str(yi) + ")")
    print("")
    print("As weighted averages:")
    print("xbar = sum(wi*xi) = " + " + ".join(wx_terms) + " = " + str(xbar))
    print("ybar = sum(wi*yi) = " + " + ".join(wy_terms) + " = " + str(ybar))

    print("")
    print("=== Composite Centroid ===")
    print("xbar = " + str(xbar))
    print("ybar = " + str(ybar))

# ---------- Symbolic mode ----------
def rect_sym():
    print("")
    print("Rectangle (reference = LOWER-LEFT corner)")
    w  = s("Width w: ")
    h  = s("Height h: ")
    x0 = s("Lower-left x0: ")
    y0 = s("Lower-left y0: ")
    A  = w + "*" + h
    cx = "(" + x0 + ")+(" + w + ")/2"
    cy = "(" + y0 + ")+(" + h + ")/2"
    return "Rectangle", A, cx, cy

def right_triangle_sym():
    print("")
    print("Right Triangle (legs from RIGHT-ANGLE corner)")
    a  = s("Leg along +x (a): ")
    b  = s("Leg along y (b): ")
    x0 = s("Right-angle corner x0: ")
    y0 = s("Right-angle corner y0: ")
    ydir = s("Y-leg direction 1=Up (+y)  2=Down (-y): ")
    A  = "1/2*" + a + "*" + b
    cx = "(" + x0 + ")+(" + a + ")/3"
    if ydir == "1":
        cy = "(" + y0 + ")+(" + b + ")/3"
        name = "Right Triangle (up)"
    else:
        cy = "(" + y0 + ")-(" + b + ")/3"
        name = "Right Triangle (down)"
    return name, A, cx, cy

def circle_sym():
    print("")
    print("Circle (reference = CENTER)")
    r  = s("Radius r: ")
    xc = s("Center xc: ")
    yc = s("Center yc: ")
    A  = "pi*" + r + "*" + r
    return "Circle", A, xc, yc

def semicircle_sym():
    print("")
    print("Semicircle (reference = CENTER of parent circle)")
    mode = s("Enter 1=Radius  or  2=Diameter: ")
    if mode == "1":
        r = s("Radius r: ")
    else:
        D = s("Diameter D: "); r = "(" + D + ")/2"
    xc = s("Center xc: ")
    yc = s("Center yc: ")
    ori = s("Bulge 1=Up 2=Down 3=Right 4=Left: ")
    A = "1/2*pi*" + r + "*" + r
    d = "(4*" + r + ")/(3*pi)"
    if   ori=="1":
        cx,cy = xc,           "(" + yc + ")+(" + d + ")"; name="Semicircle (up)"
    elif ori=="2":
        cx,cy = xc,           "(" + yc + ")-(" + d + ")"; name="Semicircle (down)"
    elif ori=="3":
        cx,cy = "(" + xc + ")+(" + d + ")", yc;          name="Semicircle (right)"
    else:
        cx,cy = "(" + xc + ")-(" + d + ")", yc;          name="Semicircle (left)"
    return name, A, cx, cy

def quarter_circle_sym():
    print("")
    print("Quarter Circle (reference = RIGHT-ANGLE corner)")
    r  = s("Radius r: ")
    x0 = s("Corner x0: ")
    y0 = s("Corner y0: ")
    q  = s("Quadrant 1=NE 2=NW 3=SW 4=SE: ")
    A = "1/4*pi*" + r + "*" + r
    d = "(4*" + r + ")/(3*pi)"
    if   q=="1":
        cx,cy = "(" + x0 + ")+(" + d + ")", "(" + y0 + ")+(" + d + ")"; name="Quarter Circle (NE)"
    elif q=="2":
        cx,cy = "(" + x0 + ")-(" + d + ")", "(" + y0 + ")+(" + d + ")"; name="Quarter Circle (NW)"
    elif q=="3":
        cx,cy = "(" + x0 + ")-(" + d + ")", "(" + y0 + ")-(" + d + ")"; name="Quarter Circle (SW)"
    else:
        cx,cy = "(" + x0 + ")+(" + d + ")", "(" + y0 + ")-(" + d + ")"; name="Quarter Circle (SE)"
    return name, A, cx, cy

def polygon_sym():
    print("")
    print("Polygon (symbolic, vertices in order CW or CCW)")
    n_str = s("Number of vertices n (>=3): ")
    try:
        n = int(n_str)
    except:
        print("Invalid n; defaulting to 3.")
        n = 3
    verts = []
    i = 1
    while i <= n:
        xi = s("  Vertex " + str(i) + " - x: ")
        yi = s("  Vertex " + str(i) + " - y: ")
        verts.append((xi, yi))
        i += 1
    terms_A2 = []
    terms_Cx = []
    terms_Cy = []
    i = 0
    while i < n:
        xi, yi = verts[i]
        xj, yj = verts[(i+1) % n]
        cross = "(({})*({}) - ({})*({}))".format(xi,yj,xj,yi)
        terms_A2.append(cross)
        terms_Cx.append("(({})+({}))*{}".format(xi,xj,cross))
        terms_Cy.append("(({})+({}))*{}".format(yi,yj,cross))
        i += 1
    A2 = " + ".join(terms_A2)
    Cx_num = " + ".join(terms_Cx)
    Cy_num = " + ".join(terms_Cy)
    A_expr  = "1/2*(" + A2 + ")"
    cx_expr = "(" + Cx_num + ")/(3*(" + A2 + "))"
    cy_expr = "(" + Cy_num + ")/(3*(" + A2 + "))"
    print("Tip: For positive area, list vertices CCW; CAS can also handle signs.")
    return "Polygon", A_expr, cx_expr, cy_expr

def add_shape_symbolic():
    print("")
    print("Add a shape (Symbolic):")
    print(" 1) Rectangle")
    print(" 2) Right Triangle (up/down)")
    print(" 3) Circle")
    print(" 4) Semicircle (radius or diameter)")
    print(" 5) Quarter Circle (quadrant)")
    print(" 6) Polygon (vertex list)")
    k = s("Choose 1-6: ")
    if   k=="1": return rect_sym()
    elif k=="2": return right_triangle_sym()
    elif k=="3": return circle_sym()
    elif k=="4": return semicircle_sym()
    elif k=="5": return quarter_circle_sym()
    else:        return polygon_sym()

def run_exercise_symbolic():
    A_terms, Ax_terms, Ay_terms = [], [], []
    i = 1
    while True:
        name, A, cx, cy = add_shape_symbolic()
        sign = s("Solid (+1) or Hole (-1)? ")
        if sign == "-1":
            A = "-1*(" + A + ")"
            name += " [HOLE]"
        add(A_terms, A)
        add(Ax_terms, A + "*(" + cx + ")")
        add(Ay_terms, A + "*(" + cy + ")")
        print("  Added #" + str(i) + ": " + name)
        more = s("Add another? 1=Yes 0=No: ")
        i += 1
        if more != "1":
            break

    SA  = " + ".join(A_terms)  if A_terms else "0"
    SAx = " + ".join(Ax_terms) if Ax_terms else "0"
    SAy = " + ".join(Ay_terms) if Ay_terms else "0"

    print("")
    print("--- Symbolic Sums ---")
    print("sumA    = " + SA)
    print("sum(Ax) = " + SAx)
    print("sum(Ay) = " + SAy)

    print("")
    print("--- Composite Centroid (unsimplified) ---")
    print("xbar = ( " + SAx + " ) / ( " + SA + " )")
    print("ybar = ( " + SAy + " ) / ( " + SA + " )")

    print("")
    print("Copy xbar,ybar into a CAS page and use simplify().")
    print("Tip: Enter polygon vertices CCW for positive area (or rely on CAS signs).")

# ---------- Main ----------
def main():
    print("=== Composite Centroid Calculator (MicroPython, ASCII) ===")
    print("Choose mode:  1=Numeric (verbose)   2=Symbolic (expressions)")
    mode = prompt_int_in((1,2), "Mode 1/2: ")
    if mode == 1:
        while True:
            run_exercise_numeric()
            ans = prompt_choice("\nDo another exercise? 1=Yes  0=No  (or 'q' to quit): ", ["1","0","q","Q"])
            if ans in ("0","q","Q"):
                print("\nGoodbye!")
                return
    else:
        run_exercise_symbolic()
        print("\nDone. (Run again for another.)")

if __name__ == "__main__":
    main()
